# handlers/withdraw/usdt_withdraw.py

from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler, CallbackQueryHandler
from database import (
    get_user_data,
    deduct_user_balance,
    refund_user_balance,
    _get_setting as db_get_setting,
    create_pending_withdrawal,
    log_transaction_and_get_id,
    get_connection,
    _get_setting,
    update_user_field,
    deduct_referral_on_loss,add_transaction
)
import json
import random
import string
import time
from constants import AWAITING_USDT_AMOUNT, AWAITING_WALLET_ADDRESS, AWAITING_NETWORK_TYPE
from config import ADMIN_IDS, OPERATIONS_CHANNEL_ID, get_config
from handlers.admin.withdraw_approval import show_pending_withdrawals

import uuid


# ===== بدء طلب سحب USDT =====
async def withdraw_usdt_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.callback_query.answer()
    await update.callback_query.message.reply_text("💵 أدخل المبلغ المراد سحبه بالدولار الأمريكي (USDT):")
    return AWAITING_USDT_AMOUNT


# ===== استلام مبلغ USDT =====
async def receive_usdt_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    amount_text = update.message.text.strip()
    if not amount_text.replace('.', '', 1).isdigit():
        await update.message.reply_text("❌ الرجاء إدخال مبلغ صحيح (أرقام فقط).")
        return AWAITING_USDT_AMOUNT

    amount_usdt = float(amount_text)

    cfg = get_config()
    MIN_WITHDRAW_USDT = cfg.get("MINIMUM_WITHDRAWAL", {}).get("USDT", 5)

    if amount_usdt < MIN_WITHDRAW_USDT:
        await update.message.reply_text(f"❌ الحد الأدنى للسحب هو {MIN_WITHDRAW_USDT} USDT.")
        return AWAITING_USDT_AMOUNT

    exchange_rate = db_get_setting("exchange_rate", 0)
    if exchange_rate <= 0:
        await update.message.reply_text("❌ لم يتم ضبط سعر الصرف من قبل الأدمن. يرجى المحاولة لاحقًا.")
        return ConversationHandler.END

    amount_syp = round(amount_usdt * exchange_rate)
    user_id_str = str(update.message.from_user.id)

    user_data = get_user_data(user_id_str)
    if not user_data or user_data.get("balance", 0.0) < amount_syp:
        current_balance = user_data.get("balance", 0.0) if user_data else 0.0
        await update.message.reply_text(
            f"⚠️ رصيدك الحالي غير كافٍ للسحب.\n"
            f"🔹 رصيدك: {current_balance:,.0f} ل.س\n"
            f"🔹 المبلغ المطلوب: {amount_syp:,} ل.س"
        )
        return ConversationHandler.END

    context.user_data['withdraw_amount_usdt'] = amount_usdt
    context.user_data['withdraw_amount_syp'] = amount_syp
    await update.message.reply_text("💼 الرجاء إدخال عنوان محفظتك (Wallet Address):")
    return AWAITING_WALLET_ADDRESS


# ===== استلام عنوان المحفظة =====
async def receive_wallet_address(update: Update, context: ContextTypes.DEFAULT_TYPE):
    wallet_address = update.message.text.strip()
    if not wallet_address:
        await update.message.reply_text("❌ العنوان غير صالح. حاول مرة أخرى:")
        return AWAITING_WALLET_ADDRESS

    context.user_data['wallet_address'] = wallet_address
    keyboard = [
        [InlineKeyboardButton("TRC20", callback_data="network_TRC20")],
        [InlineKeyboardButton("BEP20", callback_data="network_BEP20")],
        [InlineKeyboardButton("ERC20", callback_data="network_ERC20")]
    ]
    await update.message.reply_text(
        "🌐 اختر شبكة السحب:",
        reply_markup=InlineKeyboardMarkup(keyboard)
    )
    return AWAITING_NETWORK_TYPE


# ===== استلام نوع الشبكة =====
# في دالة receive_network_type في usdt_withdraw.py
async def receive_network_type(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    network = query.data.split('_', 1)[1]
    context.user_data['network_type'] = network

    user_id = query.from_user.id
    user_id_str = str(user_id)
    amount_usdt = context.user_data['withdraw_amount_usdt']  # هذا المبلغ بـ USDT
    amount_syp = context.user_data['withdraw_amount_syp']    # هذا المكافئ بالليرة
    wallet_address = context.user_data['wallet_address']

    exchange_rate = db_get_setting("exchange_rate", 0)
    if exchange_rate <= 0:
        await query.message.reply_text("❌ لم يتم ضبط سعر الصرف.")
        return ConversationHandler.END

    # خصم الرصيد (بالليرة السورية)
    if not deduct_user_balance(user_id_str, amount_syp):
        await query.message.reply_text("❌ فشل خصم الرصيد. يرجى المحاولة لاحقًا.")
        return ConversationHandler.END

    # حساب المبلغ الصافي بعد الخصم
    fee_percentage = get_config().get("WITHDRAWAL_FEE_PERCENT", {}).get("USDT", 10)
    net_amount_usdt = round(amount_usdt * (100 - fee_percentage) / 100, 2)
    
    # ✅ **التصحيح المهم**: حساب المبلغ بالليرة بناءً على net_amount_usdt
    net_amount_syp = round(net_amount_usdt * exchange_rate)

    # ✅ **التصحيح: إنشاء طلب سحب بالعملة الأصلية الصحيحة**
    create_pending_withdrawal(
        user_id=user_id_str,
        currency="USDT",  # ✅ العملة الأصلية هي USDT
        client_code=wallet_address,
        amount=amount_usdt,  # ✅ المبلغ الأصلي بالـ USDT
        method=f"usdt_{network.lower()}",  # ✅ طريقة مفصلة
        network=network,
        wallet_address=wallet_address,
        amount_in_usdt=net_amount_usdt  # ✅ المبلغ الصافي بعد الخصم
    )

    # ✅ **التصحيح: تسجيل المعاملة مع المعلومتين**
    log_transaction_and_get_id({
        "user_id": user_id_str,
        "type": "withdraw",
        "method": f"usdt_{network.lower()}",
        "amount": amount_usdt,  # ✅ المبلغ الأصلي
        "requested_amount": amount_usdt,  # ✅ المبلغ المطلوب
        "sent_amount": net_amount_usdt,  # ✅ المبلغ الصافي
        "fee_percent": fee_percentage,
        "client_number": wallet_address,
        "status": "pending",
        "extra": json.dumps({
            "network": network,
            "amount_usdt_original": amount_usdt,  # ✅ المبلغ الأصلي بالـ USDT
            "amount_syp_deducted": amount_syp,     # ✅ المبلغ المخصوم بالليرة
            "net_amount_usdt": net_amount_usdt,    # ✅ المبلغ الصافي بالـ USDT
            "net_amount_syp": net_amount_syp,      # ✅ المكافئ الصافي بالليرة
            "exchange_rate": exchange_rate
        })
    })

    # إرسال إشعار الإدارة
    text = (
        f"📤 <b>طلب سحب USDT جديد</b>\n\n"
        f"👤 المستخدم: <code>{user_id_str}</code>\n"
        f"💵 المبلغ المطلوب: {amount_usdt} USDT\n"
        f"💰 المبلغ المخصوم: {amount_syp:,} ل.س\n"
        f"📉 بعد خصم {fee_percentage}%: {net_amount_usdt} USDT ({net_amount_syp:,} ل.س)\n"
        f"🌐 الشبكة: {network}\n"
        f"💳 المحفظة: <code>{wallet_address}</code>\n"
        f"🕒 التاريخ: {time.strftime('%Y-%m-%d %H:%M:%S')}"
    )

    if OPERATIONS_CHANNEL_ID:
        try:
            await context.bot.send_message(chat_id=OPERATIONS_CHANNEL_ID, text=text, parse_mode='HTML')
        except Exception as e:
            print(f"❌ خطأ في إرسال إشعار سحب USDT للقناة: {e}")

    await query.message.reply_text(
        f"⏳ تم إرسال طلبك إلى الإدارة بانتظار الموافقة.\n"
        f"💰 سيتم تحويل {net_amount_usdt} USDT بعد خصم {fee_percentage}%.\n\n"
        f"⚠️ تم خصم {amount_syp:,} ل.س من رصيدك."
    )
    return ConversationHandler.END




def register_usdt_withdraw_handlers(dispatcher):
    dispatcher.add_handler(CallbackQueryHandler(handle_usdt_withdraw_decision, pattern=r"^(approve|reject)_withdraw_usdt:"))


async def handle_usdt_withdraw_decision(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """معالجة قرار الأدمن لطلبات سحب USDT."""
    query = update.callback_query
    await query.answer()

    try:
        # استخدام نفس نمط callback data
        action, request_id = query.data.split(':', 1)
    except Exception:
        await query.edit_message_text("❌ خطأ في تنسيق بيانات الطلب.")
        return

    # جلب طلب USDT من pending_withdrawals
    pending_request = None
    try:
        with get_connection() as conn:
            cur = conn.cursor()
            cur.execute("""
                SELECT * FROM pending_withdrawals 
                WHERE request_id = ? AND status = 'pending'
            """, (request_id,))
            row = cur.fetchone()
            if row:
                pending_request = dict(row)
    except Exception as e:
        print(f"❌ خطأ في جلب طلب USDT: {e}")

    if not pending_request:
        await query.edit_message_text("❌ هذا الطلب غير موجود أو تم معالجته.")
        return

    user_id_str = pending_request.get('user_id', '')
    original_currency = pending_request.get('original_currency', 'USDT')
    amount_usdt = float(pending_request.get('requested_amount', 0))
    amount_in_usdt = float(pending_request.get('amount_in_usdt', 0))
    wallet_address = pending_request.get('wallet_address') or pending_request.get('client_number', '')
    network = pending_request.get('network', 'Unknown')
    
    # جلب بيانات المستخدم
    user_data = get_user_data(user_id_str)
    if not user_data:
        await query.edit_message_text("❌ المستخدم غير موجود.")
        # حذف الطلب
        try:
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("DELETE FROM pending_withdrawals WHERE request_id = ?", (request_id,))
                conn.commit()
        except Exception as e:
            print(f"❌ خطأ في حذف طلب مستخدم غير موجود: {e}")
        return

    current_balance = user_data.get("balance", 0.0)
    
    # حساب سعر الصرف
    exchange_rate = _get_setting("exchange_rate", 4500)
    
    # ✅ **التصحيح: حساب المبلغ المخصوم بالليرة**
    # استخدم amount_in_usdt إذا موجود، وإلا استخدم amount_usdt
    usdt_amount = amount_in_usdt if amount_in_usdt > 0 else amount_usdt
    amount_syp_deducted = usdt_amount * exchange_rate

    original_message = query.message.text

    if action == "approve_withdraw_usdt":
        try:
            # ✅ الموافقة: تحديث حالة الطلب فقط (الرصيد تم خصمه مسبقاً)
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE pending_withdrawals 
                    SET status = 'approved' 
                    WHERE request_id = ?
                """, (request_id,))
                conn.commit()

            # ✅ تحديث حالة has_withdrawn للمستخدم
            update_user_field(user_id_str, "has_withdrawn", 1)
            
            # ✅ خصم أرباح الإحالة

            # ✅ إرسال رسالة للمستخدم
            user_msg = (
                f"✅ تمت الموافقة على طلب سحب USDT وتم تحويل المبلغ بنجاح!\n\n"
                f"💵 المبلغ: {usdt_amount} USDT\n"
                f"💰 ما يعادل: {amount_syp_deducted:,.0f} ل.س\n"
                f"🌐 الشبكة: {network}\n"
                f"💳 المحفظة: <code>{wallet_address}</code>\n\n"
                f"💼 رصيدك الحالي: {current_balance:,.0f} ل.س"
            )
            
            # ✅ إشعار الأدمن
            admin_msg = (
                f"✅ تمت الموافقة على سحب USDT.\n"
                f"👤 المستخدم: {user_id_str}\n"
                f"💵 المبلغ: {usdt_amount} USDT\n"
                f"💰 بالليرة: {amount_syp_deducted:,.0f} ل.س\n"
                f"🌐 الشبكة: {network}"
            )
            
        except Exception as e:
            print(f"❌ خطأ في معالجة موافقة USDT: {e}")
            await query.edit_message_text("❌ خطأ في معالجة الموافقة.")
            return

    elif action == "reject_withdraw_usdt":
        try:
            # ✅ **الرفض: استرداد المبلغ بالليرة السورية**
            amount_to_refund = amount_syp_deducted
            new_balance = current_balance + amount_to_refund
            
            # ✅ تحديث رصيد المستخدم
            update_user_field(user_id_str, "balance", new_balance)
            
            # ✅ تحديث حالة الطلب
            with get_connection() as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE pending_withdrawals 
                    SET status = 'rejected' 
                    WHERE request_id = ?
                """, (request_id,))
                conn.commit()
            
            # ✅ تسجيل معاملة الإرجاع
            add_transaction(
                user_id=user_id_str,
                tx_type="refund",
                amount=amount_to_refund,
                method="usdt_rejected",
                status="success",
                client_number=wallet_address
            )
            
            # ✅ رسالة للمستخدم
            user_msg = (
                f"❌ تم رفض طلب سحب USDT.\n\n"
                f"💵 المبلغ: {usdt_amount} USDT\n"
                f"💰 تم إعادة: {amount_to_refund:,.0f} ل.س\n\n"
                f"💼 رصيدك الحالي: {new_balance:,.0f} ل.س"
            )
            
            # ✅ إشعار الأدمن
            admin_msg = (
                f"❌ تم رفض سحب USDT.\n"
                f"👤 المستخدم: {user_id_str}\n"
                f"💵 المبلغ: {usdt_amount} USDT\n"
                f"💰 تم إعادة: {amount_to_refund:,.0f} ل.س"
            )
            
        except Exception as e:
            print(f"❌ خطأ في معالجة رفض USDT: {e}")
            await query.edit_message_text("❌ خطأ في معالجة الرفض.")
            return

    else:
        return

    # إرسال الرسائل
    try:
        await context.bot.send_message(chat_id=int(user_id_str), text=user_msg, parse_mode='HTML')
    except Exception as e:
        print(f"⚠️ فشل إرسال إشعار للمستخدم {user_id_str}: {e}")

    # تحديث رسالة الأدمن
    await query.edit_message_text(original_message + f"\n\n{admin_msg}")
    
    # تحديث قائمة السحوبات
    await show_pending_withdrawals(update, context)