from telegram import InlineKeyboardButton, InlineKeyboardMarkup, Update
from telegram.ext import ContextTypes, ConversationHandler
from database import (
    get_user_balance,
    deduct_user_balance,
    create_pending_withdrawal,
    add_transaction,
    refund_user_balance
)
from config import get_config, ADMIN_IDS, OPERATIONS_CHANNEL_ID
import time

from constants import AWAITING_SYRIATEL_CODE, AWAITING_SYRIATEL_AMOUNT

# --- زر الإلغاء ---
def get_cancel_button():
    return [InlineKeyboardButton("❌ إلغاء العملية", callback_data="cancel_withdraw_syriatel")]

CANCEL_NOTE = "\n\n⚠️ <b>ملاحظة:</b> يرجى إلغاء العملية قبل الضغط على أي زر آخر لتجنب تعارض الطلبات."

# --- إلغاء العملية ---
async def cancel_withdraw_syriatel(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    context.user_data.clear()
    await query.edit_message_text("❌ تم إلغاء عملية السحب من قبل العميل.", parse_mode='HTML')
    return ConversationHandler.END

# --- بدء العملية ---
async def withdraw_syriatel_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    context.user_data.clear()
    context.user_data["is_withdrawing"] = True
    query = update.callback_query
    await query.answer()

    await query.edit_message_text(
        "🔑 الرجاء إدخال رقم الهاتف الخاص بالعميل.\nمثال: 0912345678" + CANCEL_NOTE,
        reply_markup=InlineKeyboardMarkup([get_cancel_button()]),
        parse_mode='HTML'
    )
    return AWAITING_SYRIATEL_CODE

# --- استلام رقم الهاتف ---
async def receive_syriatel_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    phone_number = update.message.text.strip()
    if not (phone_number.isdigit() and len(phone_number) == 10 and phone_number.startswith("09")):
        await update.message.reply_text("❌ رقم الهاتف غير صالح. الرجاء إدخال رقم يبدأ بـ 09 ويتكون من 10 أرقام.")
        return AWAITING_SYRIATEL_CODE

    context.user_data["syriatel_number"] = phone_number
    await update.message.reply_text(
        "💰 الآن أرسل المبلغ الذي تريد سحبه (بالأرقام فقط):" + CANCEL_NOTE,
        reply_markup=InlineKeyboardMarkup([get_cancel_button()]),
        parse_mode='HTML'
    )
    return AWAITING_SYRIATEL_AMOUNT

# --- استلام المبلغ وتنفيذ السحب يدوي ---
async def receive_syriatel_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user_id = str(update.effective_user.id)
    phone_number = context.user_data.get("syriatel_number")
    if not phone_number:
        await update.message.reply_text("❌ لم يتم تحديد رقم العميل، يرجى البدء من جديد.")
        return ConversationHandler.END

    config = get_config()
    MIN_WITHDRAW = config.get("MINIMUM_WITHDRAWAL", {}).get("SYRIATEL", 50_000)
    MAX_WITHDRAW = config.get("MAX_DIRECT_WITHDRAW", {}).get("SYRIATEL", 500_000)
    FEE_PERCENT = config.get("WITHDRAWAL_FEE_PERCENT", {}).get("SYRIATEL", 5)

    try:
        amount = int(update.message.text.strip())
        if amount <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ مبلغ غير صالح. الرجاء إرسال رقم موجب صحيح.")
        return AWAITING_SYRIATEL_AMOUNT

    if amount < MIN_WITHDRAW:
        await update.message.reply_text(f"❌ الحد الأدنى للسحب هو {MIN_WITHDRAW:,} ل.س.")
        return AWAITING_SYRIATEL_AMOUNT

    current_balance = get_user_balance(user_id)
    if amount > current_balance:
        await update.message.reply_text(f"❌ لا يمكنك سحب {amount:,} ل.س، رصيدك الحالي {current_balance:,} ل.س.")
        return AWAITING_SYRIATEL_AMOUNT

    # خصم الرصيد مباشرة
   

    # حساب الرسوم
    fee_amount = int(amount * FEE_PERCENT / 100)
    amount_after_fee = amount - fee_amount

    # إنشاء طلب سحب معلق يدوي
    request_id = create_pending_withdrawal(
        user_id=user_id,
        client_code=phone_number,
        amount=amount,
        method="syriatel",
        currency="SYP"
    )

    if not request_id:
        refund_user_balance(user_id, amount)
        await update.message.reply_text("❌ فشل في إنشاء طلب السحب. تم إعادة الرصيد.")
        return ConversationHandler.END

    # إضافة سجل المعاملة
    try:
        add_transaction(
            user_id=user_id,
            tx_type="withdraw",
            amount=amount,
            method="syriatel_pending",
            status="pending",
            client_number=phone_number
        )
    except Exception as e:
        print(f"⚠️ فشل إضافة سجل المعاملة: {e}")

    # إشعار المستخدم
    await update.message.reply_text(
        f"✅ تم تقديم طلب سحب سيرياتيل بنجاح!\n\n"
        f"• المبلغ المطلوب: {amount:,} ل.س\n"
        f"• قيمة الرسوم: {fee_amount:,} ل.س\n"
        f"• المبلغ بعد الخصم: {amount_after_fee:,} ل.س\n"
        f"• رقم العميل: {phone_number}\n"
        f"• حالة الطلب: ⏳ قيد المراجعة من قبل الإدارة",
        parse_mode="HTML"
    )

    # إشعار الإدارة / القناة
    msg = (
        f"🆕 <b>طلب سحب سيرياتيل جديد:</b>\n"
        f"👤 المستخدم: <code>{user_id}</code>\n"
        f"💰 المبلغ المطلوب: {amount:,} ل.س\n"
        f"• قيمة الرسوم: {fee_amount:,} ل.س\n"
        f"• المبلغ بعد الخصم: {amount_after_fee:,} ل.س\n"
        f"📞 رقم العميل: <code>{phone_number}</code>\n"
        f"📌 الحالة: ⏳ يحتاج مراجعة يدوية\n"
        f"🔗 معرف الطلب: <code>{request_id}</code>"
    )

    keyboard = [
        [
            InlineKeyboardButton("✅ الموافقة", callback_data=f"approve_withdraw:{request_id}"),
            InlineKeyboardButton("❌ الرفض", callback_data=f"reject_withdraw:{request_id}")
        ]
    ]

    for admin_id in ADMIN_IDS:
        try:
            await context.bot.send_message(
                chat_id=admin_id,
                text=msg,
                parse_mode="HTML",
                reply_markup=InlineKeyboardMarkup(keyboard)
            )
        except Exception as e:
            print(f"⚠️ فشل إرسال إشعار للأدمن {admin_id}: {e}")

    if OPERATIONS_CHANNEL_ID:
        try:
            await context.bot.send_message(
                chat_id=OPERATIONS_CHANNEL_ID,
                text=msg,
                parse_mode="HTML"
            )
        except Exception as e:
            print(f"⚠️ فشل إرسال إشعار للقناة: {e}")

    context.user_data.clear()
    return ConversationHandler.END


# ========== إلغاء العملية ==========
async def cancel_withdraw_client(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ تم حذف db = load_db()
    if "is_withdrawing" in context.user_data:
        del context.user_data["is_withdrawing"]

    query = update.callback_query
    await query.answer()
    await query.edit_message_text("❌ تم إلغاء عملية السحب من قبل العميل.")
    return ConversationHandler.END