from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler

from config import ADMIN_IDS
import time
import uuid

# الحالات
from constants import (
    AWAITING_HAWALA_AMOUNT,
    AWAITING_HAWALA_NUMBER,
    AWAITING_HAWALA_NAME,
    AWAITING_HAWALA_PROOF
)
from handlers.wallet.transaction_history import send_transaction_to_channel

MIN_HAWALA_WITHDRAW = 50000


# ===== زر الإلغاء =====
def get_cancel_button():
    return [[InlineKeyboardButton("❌ إلغاء العملية", callback_data="cancel_withdraw_client")]]


# ===== بدء عملية السحب =====
async def withdraw_hawala_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    # ← إذا كان المستخدم في عملية سحب جارية، نلغيها أولًا
    if context.user_data.get("is_withdrawing"):
        await query.edit_message_text(
            "❌ تم إلغاء العملية السابقة تلقائيًا بسبب الضغط على زر جديد."
        )
        context.user_data.clear()
        return ConversationHandler.END

    context.user_data["is_withdrawing"] = True

    await query.edit_message_text(
        "💰 الرجاء إدخال المبلغ الذي تريد سحبه عبر الحوالة (هرم / فؤاد):",
        reply_markup=InlineKeyboardMarkup(get_cancel_button())
    )
    return AWAITING_HAWALA_AMOUNT


# ===== استلام المبلغ =====
async def receive_hawala_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        amount = int(update.message.text.strip())
        if amount < MIN_HAWALA_WITHDRAW:
            await update.message.reply_text(f"❌ الحد الأدنى للسحب هو {MIN_HAWALA_WITHDRAW:,} ل.س.")
            return AWAITING_HAWALA_AMOUNT
    except ValueError:
        await update.message.reply_text("❌ الرجاء إدخال مبلغ صالح.")
        return AWAITING_HAWALA_AMOUNT

    context.user_data["hawala_amount"] = amount
    await update.message.reply_text(
        "📞 الآن الرجاء إدخال رقم الهاتف الذي سيتم استلام الحوالة عليه:",
        reply_markup=InlineKeyboardMarkup(get_cancel_button())
    )
    return AWAITING_HAWALA_NUMBER


# ===== استلام الرقم =====
async def receive_hawala_number(update: Update, context: ContextTypes.DEFAULT_TYPE):
    phone = update.message.text.strip()
    if not (phone.isdigit() and len(phone) == 10 and phone.startswith("09")):
        await update.message.reply_text("❌ الرجاء إدخال رقم هاتف صالح يبدأ بـ 09.")
        return AWAITING_HAWALA_NUMBER

    context.user_data["hawala_phone"] = phone
    await update.message.reply_text(
        "🧾 الآن الرجاء إدخال اسمك الثلاثي كما سيظهر في الحوالة:",
        reply_markup=InlineKeyboardMarkup(get_cancel_button())
    )
    return AWAITING_HAWALA_NAME


# ===== استلام الاسم الكامل =====
async def receive_hawala_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
    full_name = update.message.text.strip()
    if len(full_name.split()) < 3:
        await update.message.reply_text("❌ الرجاء إدخال الاسم الثلاثي الكامل.")
        return AWAITING_HAWALA_NAME

    user_id = str(update.effective_user.id)
    amount = context.user_data["hawala_amount"]
    phone = context.user_data["hawala_phone"]

    request_id = str(uuid.uuid4())
    db.setdefault("pending_hawala_withdraws", {})[request_id] = {
        "user_id": user_id,
        "amount": amount,
        "phone": phone,
        "full_name": full_name,
        "timestamp": time.time(),
        "status": "pending"
    }
    save_all_changes()

    # إرسال الطلب للأدمن
    keyboard = InlineKeyboardMarkup([[
        InlineKeyboardButton("✅ تم التحويل", callback_data=f"hawala_approve:{request_id}"),
        InlineKeyboardButton("❌ رفض", callback_data=f"hawala_reject:{request_id}")
    ]])

    await update.message.reply_text("✅ تم إرسال طلبك للإدارة. يرجى انتظار المراجعة.")

    admin_msg = (
        f"📦 <b>طلب سحب جديد عبر حوالة</b>\n"
        f"👤 المستخدم: <code>{update.effective_user.full_name}</code> (<code>{user_id}</code>)\n"
        f"💰 المبلغ: <code>{amount:,}</code> ل.س\n"
        f"📞 الرقم: <code>{phone}</code>\n"
        f"🧾 الاسم: <code>{full_name}</code>\n\n"
        f"الرجاء اختيار إجراء:"
    )

    for admin_id in ADMIN_IDS:
        try:
            await context.bot.send_message(
                chat_id=admin_id,
                text=admin_msg,
                parse_mode="HTML",
                reply_markup=keyboard
            )
        except Exception as e:
            print(f"⚠️ فشل إرسال إشعار للأدمن {admin_id}: {e}")


    context.user_data.clear()
    return ConversationHandler.END


# ===== رد الأدمن - تم التحويل =====
async def admin_approve_hawala(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    request_id = query.data.split(":")[1]
    request = db.get("pending_hawala_withdraws", {}).get(request_id)

    if not request:
        await query.edit_message_text("❌ لم يتم العثور على هذا الطلب.")
        return

    user_id = request["user_id"]
    await query.edit_message_text(
        "✅ الرجاء الآن إرسال <b>صورة إشعار التحويل</b> لإرسالها للعميل.",
        parse_mode="HTML"
    )

    context.user_data["hawala_proof_request"] = {
        "user_id": user_id,
        "request_id": request_id
    }

    return AWAITING_HAWALA_PROOF


# ===== استلام صورة الإشعار من الأدمن =====
async def receive_hawala_proof(update: Update, context: ContextTypes.DEFAULT_TYPE):
    proof_data = context.user_data.get("hawala_proof_request")
    if not proof_data:
        await update.message.reply_text("❌ لا يوجد طلب مرتبط بهذه الصورة.")
        return ConversationHandler.END

    photo_id = update.message.photo[-1].file_id
    user_id = proof_data["user_id"]
    request_id = proof_data["request_id"]

    withdraws = db.get("pending_hawala_withdraws", {})
    request = withdraws.get(request_id)
    if not request:
        await update.message.reply_text("⚠️ الطلب غير موجود في قاعدة البيانات.")
        return ConversationHandler.END

    # ✅ تحديث حالة الطلب
    request["status"] = "completed"
    request["proof_photo"] = photo_id
    save_all_changes()
    # --- منع ربح الإحالة بسبب السحب عبر حوالة ---
    referred_id = str(user_id)  # المستخدم الذي سحب
    referrer_id = db['referral_relationships'].get(referred_id)
    if referrer_id:
        db.setdefault('referral_earnings', {}).setdefault(referrer_id, {})
        db['referral_earnings'][referrer_id]['blocked_until'] = int(time.time()) + 10*24*60*60
        print(f"⏱ تم حظر نسبة الداعي {referrer_id} لمدة 10 أيام بسبب سحب المدعو {referred_id} عبر حوالة")
        save_all_changes()


    # ✅ إنشاء سجل العملية
    new_transaction = {
        "user_id": str(user_id),
        "type": "withdraw",
        "amount": request["amount"],
        "method": "حوالة (هرم / فؤاد)",
        "timestamp": time.time(),
        "status": "success",
        "proof_photo": photo_id
    }

    db.setdefault("transaction_logs", []).append(new_transaction)
    save_all_changes()

    # ✅ إرسالها إلى القناة
    await send_transaction_to_channel(new_transaction, context)

    # ✅ إرسال الصورة للعميل
    try:
        await context.bot.send_photo(
            chat_id=user_id,
            photo=photo_id,
            caption=(
                f"✅ تم تحويل المبلغ بنجاح عبر الحوالة (هرم / فؤاد).\n\n"
                f"💰 المبلغ: <code>{request['amount']:,}</code> ل.س\n"
                f"📞 الرقم: <code>{request['phone']}</code>\n"
                f"🧾 الاسم: <code>{request['full_name']}</code>"
            ),
            parse_mode="HTML"
        )
    except Exception as e:
        print(f"⚠️ فشل إرسال الإشعار للعميل: {e}")

    await update.message.reply_text("📤 تم إرسال صورة الإشعار إلى العميل وتم حفظ العملية في السجل.")
    context.user_data.clear()
    return ConversationHandler.END


# ===== رفض الأدمن =====
async def admin_reject_hawala(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()

    request_id = query.data.split(":")[1]
    request = db.get("pending_hawala_withdraws", {}).get(request_id)
    if not request:
        await query.edit_message_text("❌ الطلب غير موجود.")
        return

    user_id = request["user_id"]

    # تحديث الحالة
    db["pending_hawala_withdraws"][request_id]["status"] = "rejected"
    save_all_changes()

    # إنشاء سجل مرفوض
    rejected_transaction = {
        "user_id": str(user_id),
        "type": "withdraw",
        "amount": request["amount"],
        "method": "حوالة (هرم / فؤاد)",
        "timestamp": time.time(),
        "status": "failed"
    }

    db.setdefault("transaction_logs", []).append(rejected_transaction)
    save_all_changes()

    # إرسالها إلى القناة
    await send_transaction_to_channel(rejected_transaction, context)

    # إعلام المستخدم
    try:
        await context.bot.send_message(user_id, "❌ تم رفض طلب السحب عبر الحوالة من قبل الإدارة.")
    except Exception:
        pass

    await query.edit_message_text("🚫 تم رفض الطلب.")


# ===== إلغاء العملية =====
async def cancel_withdraw_client(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if "is_withdrawing" in context.user_data:
        del context.user_data["is_withdrawing"]

    query = update.callback_query
    await query.answer()
    await query.edit_message_text("❌ تم إلغاء عملية السحب من قبل العميل.")
    context.user_data.clear()
    return ConversationHandler.END
