import datetime
import asyncio
import sqlite3
from database import (
    get_user_data,
    get_user_net_balance_from_transactions,
    get_referral_relationships,
    get_user_referral_earnings,
    insert_referral_earning,
    get_user_referral_summary,
    update_user_referral_summary,
    has_minimum_active_referrals,
    is_user_active,
    get_last_user_deposit,
    get_connection  # تأكد من وجود هذه الدالة في database.py
)
from config import get_config


async def hourly_referral_update(context):
    print("🔄 بدء تحديث أرباح الإحالة الذكي (منع التكرار)...")
    cfg = get_config()
    ref_cfg = cfg.get("REFERRAL_PERCENTAGES", {})
    REF_PERCENTAGES = [
        float(ref_cfg.get("LEVEL_1", 5)) / 100,
        float(ref_cfg.get("LEVEL_2", 2)) / 100,
        float(ref_cfg.get("LEVEL_3", 1)) / 100
    ]

    relationships_dict = get_referral_relationships()
    relationships = list(relationships_dict.items())
    
    updated = 0
    skipped = 0

    for invited_id, referrer_id in relationships:
        # 1. التحقق من نشاط المستخدم
        if not is_user_active(invited_id):
            skipped += 1
            continue

        # 2. جلب آخر إيداع
        latest_deposit = get_last_user_deposit(invited_id)
        if not latest_deposit:
            skipped += 1
            continue

        deposit_amount = latest_deposit["amount"]
        deposit_tx_id = latest_deposit["transaction_id"]
        if deposit_amount > 125_000:
            skipped += 1
            continue
        # 3. معالجة مستويات الإحالة صعوداً
        current_invited = invited_id
        
        for level, percent in enumerate(REF_PERCENTAGES, start=1):
            target_referrer = relationships_dict.get(current_invited)
            
            if not target_referrer:
                break

            # صياغة المعرف الفريد للربح (هذا هو المفتاح لمنع التكرار)
            earning_id = f"lvl{level}_{deposit_tx_id}_{target_referrer}"

            # --- الخطوة الأهم: فحص هل الربح موجود مسبقاً؟ ---
            try:
                with get_connection() as conn:
                    cur = conn.cursor()
                    cur.execute("SELECT status FROM referral_earnings WHERE id = ?", (earning_id,))
                    exists = cur.fetchone()
                
                if exists:
                    # إذا وجد السجل، نتخطاه فوراً (سواء كان pending أو distributed)
                    # هذا يمنع الدالة من مسح حالة 'distributed' التي وضعها الأدمن
                    current_invited = target_referrer
                    continue
            except Exception as e:
                print(f"⚠️ خطأ في فحص وجود السجل: {e}")
                current_invited = target_referrer
                continue

            # 4. فحص شرط الحد الأدنى للإحالات
            if not has_minimum_active_referrals(target_referrer, minimum=3):
                current_invited = target_referrer
                continue

            # 5. حساب الربح
            current_earning = round(deposit_amount * percent, 2)
            if current_earning <= 0:
                current_invited = target_referrer
                continue

            # 6. إضافة الربح الجديد فقط
            try:
                # حذفنا سطر delete_referral_earning لأنه هو سبب المشكلة
                
                insert_referral_earning({
                    "earning_id": earning_id,
                    "referrer_id": target_referrer,
                    "invited_id": invited_id,
                    "amount": current_earning,
                    "status": "pending",
                    "level": level,
                    "created_at": datetime.datetime.now().isoformat()
                })

                # 7. تحديث ملخص الإحالة (Summary) للمحيل بناءً على القيم الفعلية في الجدول
                all_earnings = get_user_referral_earnings(target_referrer)
                total_pending = sum(e['amount'] for e in all_earnings if e['status'] == 'pending')
                total_available = sum(e['amount'] for e in all_earnings if e['status'] == 'available')

                update_user_referral_summary(
                    target_referrer,
                    total_pending,
                    total_available,
                    total_pending + total_available,
                    last_processed_deposit_id=deposit_tx_id
                )
                
                updated += 1
                print(f"✅ سجل جديد: {current_earning} ل.س للمحيل {target_referrer}")

            except Exception as e:
                print(f"❌ فشل تحديث أرباح المحيل {target_referrer}: {e}")

            current_invited = target_referrer

    print(f"📊 [النتيجة] تم إضافة {updated} سجل جديد، وتخطى {skipped} مستخدم.")