# handlers/ishansy/website_deposit.py
import asyncio
from telegram import Update
from telegram.ext import ContextTypes, ConversationHandler
# 🟢 استيراد دوال SQLite الجديدة
from database import (
    get_user_data, 
    add_transaction, 
    add_user_history,
    update_user_field, # سنستخدمها لتحديث رصيد البونص بشكل مباشر
    # ❌ تم حذف db, save_all_changes, load_db
)
from ishansy_controller import IshansyController
from constants import AWAITING_WEBSITE_DEPOSIT_AMOUNT
from global_state import activate_global_lock
# ⬅️ دالة مساعدة متزامنة لتشغيل عمليات Ishansy في خيط منفصل
def _run_ishansy_deposit_operation(target_username: str, amount: float) -> tuple[bool, str]:
    """
    تنفيذ عملية الإيداع باستخدام الـ API الرسمية فقط.
    بدون كوكيز – بدون كابتشا – بدون متصفح.
    """

    controller = IshansyController(username="luffy@gmail.com", password="Aa@723723")

    try:
        # 1️⃣ تسجيل دخول API
        success, resp = controller.login()
        if not success:
            return False, f"❌ فشل تسجيل الدخول: {resp}"

        # 2️⃣ البحث عن اللاعب للحصول على playerId
        data = controller.get_player(username=target_username)
        if not data or not data.get("status"):
            return False, "❌ فشل جلب معلومات اللاعب من السيرفر"

        # ✅ استخدام 'records' وليس 'items'
        records = data.get("result", {}).get("records", [])
        if not records:
            return False, "❌ اسم اللاعب غير موجود في النظام"

        player_id = records[0]["playerId"]  # المفتاح الجديد

        # 3️⃣ إجراء الإيداع
        result = controller.deposit(player_id=player_id, amount=amount)

        if result.get("status") is True:
            return True, f"✅ تم إيداع {amount:,.0f} بنجاح للمستخدم {target_username}"

        return False, f"❌ فشل الإيداع: {result}"

    except Exception as e:
        return False, f"❌ حدث خطأ تقني غير متوقع: {e}"

async def _execute_deposit_in_background(user_id_str: str, target_username: str, amount_to_deposit: float, is_bonus: bool, bot, chat_id_to_reply: int, message_to_edit: int , user_data):
    await activate_global_lock(3)
    """
    تنفيذ عملية الإيداع على Ishansy في خيط منفصل وحفظ البيانات وإرسال الرسالة النهائية.
    """
    
    # 1. تنفيذ العملية المتزامنة في خيط منفصل (الاتصال بـ API)
    try:
        success, msg = await asyncio.to_thread(_run_ishansy_deposit_operation, target_username, amount_to_deposit)
    except Exception as e:
        success = False
        msg = f"❌ خطأ غير متوقع أثناء الشحن الخارجي: {e}"

    # 2. تحديث قاعدة البيانات وإعداد الرسالة النهائية
    if success:
        # 🟢 جلب البيانات المحدثة للحساب للخصم
        account_data = get_user_data(user_id_str)
        
        # 🟢 تحديث الرصيد بناءً على نوع الإيداع
        if is_bonus:
            source_text = "رصيد الهدايا"
            
            # 🟢 التعديل: خصم من bonus_balance
            new_bonus = account_data.get('bonus_balance', 0.0) - amount_to_deposit
            update_user_field(user_id_str, 'bonus_balance', new_bonus)
            
            # تسجيل المعاملة
            add_transaction(
                user_id=user_id_str, 
                tx_type="bonus_to_website", 
                amount=amount_to_deposit,
                method="website_deposit", 
                status="success",
                client_number=target_username
            )
            
            # تسجيل العملية في سجل المستخدم (user_history)
            add_user_history(user_id_str, "bonus_to_website", amount_to_deposit, "website_deposit")
            
        else:
            source_text = "رصيدك الأساسي"
            
            # 🟢 التعديل: خصم من الرصيد الأساسي أولاً
            current_balance = account_data.get('balance', 0.0)
            
            
            # خصم الرصيد الأساسي
            
            
            # تسجيل المعاملة كـ "withdraw" أو "website_deposit"
            add_transaction(
                user_id=user_id_str, 
                tx_type="website_deposit", 
                amount=amount_to_deposit,
                method="balance_to_site", 
                status="success",
                client_number=target_username
            )
            
            # تسجيل العملية في سجل المستخدم (user_history)
            add_user_history(user_id_str, "website_deposit", amount_to_deposit, "balance_to_site")

        # استخدام التنسيق Markdown للأرقام
        final_msg = f"✅ تم شحن `{int(amount_to_deposit):,}` ل.س من {source_text} إلى حسابك في الموقع بنجاح!"
    else:
        # ❌ منطق التعامل مع رسالة الخطأ
        MAX_ERROR_LENGTH = 1000
        
        if len(msg) > MAX_ERROR_LENGTH:
            shortened_msg = msg[:MAX_ERROR_LENGTH] + "\n\n... (تم اقتطاع بقية رسالة الخطأ لطولها)"
        else:
            shortened_msg = msg
            
        final_msg = f"❌ فشلت عملية الإيداع.\n\n🟥 السبب:\n{shortened_msg}"

    # 3. إرسال الرسالة النهائية للمستخدم (تعديل الرسالة الأولية)
    try:
        await bot.edit_message_text(
            chat_id=chat_id_to_reply,
            message_id=message_to_edit,
            text=final_msg,
            parse_mode='Markdown'
        )
    except Exception:
        await bot.send_message(
            chat_id=chat_id_to_reply, 
            text=final_msg,
            parse_mode='Markdown'
        )
    finally:
        # ✅ إزالة القفل مهما كانت النتيجة
        user_data.pop('deposit_lock', None)
          
async def deposit_bonus_balance(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    user_id_str = str(query.from_user.id)
    if context.user_data.get('deposit_lock'):
        await query.message.reply_text("⚠️ جاري تنفيذ عملية الإيداع حاليًا. لا يمكنك البدء مرة أخرى.")
        return

    # ✅ تفعيل القفل
    context.user_data['deposit_lock'] = True

    # 🟢 التعديل: استخدام get_user_data
    account = get_user_data(user_id_str)
    if not account:
        await query.message.reply_text("❌ ليس لديك حساب.")
        # ❌ إزالة القفل عند الفشل المبكر
        context.user_data.pop('deposit_lock', None) 
        return

    bonus = account.get('bonus_balance', 0.0) # تأكد أنها float
    if bonus <= 0:
        await query.message.reply_text("❌ لا يوجد رصيد هدايا للشحن.")
        # ❌ إزالة القفل عند الفشل المبكر
        context.user_data.pop('deposit_lock', None)
        return
    
    target_username = account.get('username')
    
    # 1. إرسال رسالة "جاري الانتظار" والحصول على ID
    await_message = await query.message.reply_text(f"⏳ جارٍ شحن {int(bonus):,} ل.س من رصيد الهدايا إلى الموقع. يرجى الانتظار...")
    
    # 2. إطلاق المهمة الخلفية دون انتظار
    context.application.create_task(
        _execute_deposit_in_background(
            user_id_str=user_id_str,
            target_username=target_username,
            amount_to_deposit=bonus,
            is_bonus=True, # تحديد أنها عملية رصيد هدايا
            bot=context.bot,
            chat_id_to_reply=update.effective_chat.id,
            message_to_edit=await_message.message_id,
            user_data=context.user_data
        )
    )
    # لا يوجد return ConversationHandler.END هنا لأنها ليست جزءاً من Conversation
    # فقط نتركها تنتهي بشكل طبيعي بعد إطلاق المهمة.
    
async def deposit_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    query = update.callback_query
    await query.answer()
    user_id_str = str(query.from_user.id)

    # 🟢 التعديل: استخدام get_user_data
    account = get_user_data(user_id_str)
    balance = account.get('balance', 0.0) if account else 0.0
    
    if balance <= 0:
        await query.edit_message_text("❌ لا يوجد رصيد كافي في محفظتك للإيداع.")
        return ConversationHandler.END

    await query.edit_message_text(
        f"💰 رصيدك الحالي: {int(balance):,}\n" # استخدم int للظهور
        "الرجاء إرسال المبلغ الذي تريد إيداعه في الموقع."
    )
    return AWAITING_WEBSITE_DEPOSIT_AMOUNT


async def receive_deposit_amount(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user_id_str = str(update.effective_user.id)
    
    try:
        amount_to_deposit = float(update.message.text)
        if amount_to_deposit <= 0:
            raise ValueError
    except ValueError:
        await update.message.reply_text("❌ مبلغ غير صالح. الرجاء إرسال رقم موجب صحيح.")
        return ConversationHandler.END
    MIN_DEPOSIT = 5000
    if amount_to_deposit < MIN_DEPOSIT:
        await update.message.reply_text(f"❌ الحد الأدنى للإيداع هو {MIN_DEPOSIT:,} ل.س.")
        return ConversationHandler.END

    # 🟢 التعديل: استخدام get_user_data
    user_account = get_user_data(user_id_str)

    if not user_account:
        await update.message.reply_text("❌ لم يتم العثور على حسابك.")
        return ConversationHandler.END

    # الحصول على الرصيد الحالي وتحويله إلى float للتأكد
    current_balance = float(user_account.get('balance', 0.0))
    
    if amount_to_deposit > current_balance:
        # ✅ إصلاح: استخدام int للظهور بدون فاصلة عشرية
        await update.message.reply_text(f"❌ الرصيد غير كافٍ. رصيدك الحالي: {int(current_balance):,} ل.س")
        return ConversationHandler.END

    target_username = user_account.get('username')
    
    if context.user_data.get('deposit_lock'):
        await update.message.reply_text("⚠️ جاري تنفيذ عملية الإيداع حاليًا. لا يمكنك البدء مرة أخرى.")
        return ConversationHandler.END
    
    # 1. إرسال رسالة "جاري الانتظار" والحصول على ID
    await_message = await update.message.reply_text("⏳ جارٍ تنفيذ الإيداع على الموقع، يرجى الانتظار...")
    
    # ✅ تفعيل القفل قبل إطلاق المهمة
    context.user_data['deposit_lock'] = True
    
    # 2. إطلاق المهمة الخلفية دون انتظار
    context.application.create_task(
        _execute_deposit_in_background(
            user_id_str=user_id_str,
            target_username=target_username,
            amount_to_deposit=amount_to_deposit,
            is_bonus=False, # تحديد أنها عملية رصيد أساسي
            bot=context.bot,
            chat_id_to_reply=update.effective_chat.id,
            message_to_edit=await_message.message_id,
            user_data=context.user_data
        )
    )

    # 3. إنهاء المحادثة فوراً
    return ConversationHandler.END
async def cancel_deposit(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # ❌ إزالة القفل في حال تم الإلغاء
    context.user_data.pop('deposit_lock', None) 
    
    if update.callback_query:
        query = update.callback_query
        await query.answer()
        await query.edit_message_text("❌ تم إلغاء عملية الإيداع.")
    elif update.message:
        await update.message.reply_text("❌ تم إلغاء عملية الإيداع.")
    return ConversationHandler.END