import httpx
import asyncio
import time
from datetime import datetime
from telegram import Update
from telegram.ext import ContextTypes, ConversationHandler
from config import API_BASE_URL, ARAB_PAYMENT_API_KEY
from constants import AWAITING_DEPOSIT_AMOUNT
from database import (
    get_user_data,
    create_local_user,
    add_transaction,
    _get_setting as get_config_value,
    add_user_history,
    add_admin_history,
    add_used_transaction,
    check_used_transaction,
    get_user_balance,
    update_user_field
)
from config import get_config

async def process_arab_pay_success(update, context, transaction_id, details):
    """معالجة الدفع الناجح عبر Arab Pay مع قاعدة البيانات الجديدة"""
    user = update.effective_user
    user_id = str(user.id)
    
    try:
        # 1. جلب المبلغ المدفوع (USDT)
        amount_usdt = float(details.get('amount', 0))
        if amount_usdt <= 0:
            await update.effective_message.edit_text("❌ المبلغ غير صالح.")
            context.user_data.clear()
            return

        # 2. التحقق من عدم تكرار المعاملة
        if await check_used_transaction(transaction_id):
            await update.effective_message.edit_text("⚠️ تم معالجة هذه الدفعة مسبقاً وتم إضافة الرصيد.")
            context.user_data.clear()
            return

        # 3. حساب المبلغ بالليرة السورية من قاعدة البيانات
        exchange_rate_str = get_config_value("exchange_rate")
        try:
            exchange_rate = float(exchange_rate_str) if exchange_rate_str else 4500.0
        except (ValueError, TypeError):
            exchange_rate = 4500.0
        
        amount_syp = amount_usdt * exchange_rate

        # 4. التأكد من وجود حساب المستخدم
        if not get_user_data(user_id):
            create_local_user(user_id, user.username)

        # 5. إضافة المعاملة إلى قاعدة البيانات
        transaction_success = add_transaction(
            user_id=user_id,
            tx_type="deposit",
            amount=amount_syp,
            method="USDT (Arab Pay)",
            status="success",
            client_number=transaction_id
        )
        
        if not transaction_success:
            await update.effective_message.edit_text("❌ فشل في إضافة المعاملة.")
            context.user_data.clear()
            return

        # 6. تسجيل العملية كمستخدمة لمنع التكرار
        await add_used_transaction(
            reference_id=transaction_id,
            amount=amount_syp,
            method="USDT (Arab Pay)",
            user_id=user_id
        )

        # 7. إضافة إلى سجل المستخدم
        add_user_history(user_id, "deposit", amount_syp, "USDT (Arab Pay)")

        # 8. سجل الأدمن
        add_admin_history(
            user_identifier=user_id,
            operation_type="deposit",
            amount=amount_syp,
            method="USDT (Arab Pay)",
            status="success",
            extra={
                "transaction_id": transaction_id,
                "amount_usdt": amount_usdt,
                "exchange_rate": exchange_rate
            }
        )

        # 9. الحصول على الرصيد الجديد للعرض
        new_balance = get_user_balance(user_id)
        # 10. معالجة الإحالة مباشرة (نظام جديد)
        try:
            from handlers.referral.referral_service import process_referral_for_deposit

            process_referral_for_deposit(
                user_id=user_id,
                amount=float(amount_syp),
                transaction_id=transaction_id
            )
        except Exception as e:
            print(f"⚠️ فشل معالجة الإحالة (Arab Pay): {e}")


        # 10. إعلام المستخدم
        success_msg = (
            f"✅ <b>تم الإيداع بنجاح!</b>\n\n"
            f"💵 المبلغ المستلم: <code>{amount_usdt} USDT</code>\n"
            f"💱 سعر الصرف: <code>{exchange_rate}</code>\n"
            f"💰 المبلغ المضاف لرصيدك: <code>{amount_syp:,.0f} ل.س</code>\n"
            f"💳 رصيدك الحالي: <code>{new_balance:,.0f} ل.س</code>\n\n"
            f"🆔 رقم المعاملة: <code>{transaction_id}</code>"
        )
        
        await update.effective_message.edit_text(success_msg, parse_mode='HTML')
        
        # 11. تنظيف البيانات المؤقتة
        context.user_data.clear()

    except ValueError as e:
        await update.effective_message.edit_text(f"❌ خطأ في البيانات: {str(e)}")
        context.user_data.clear()
    except Exception as e:
        await update.effective_message.edit_text(f"❌ حدث خطأ غير متوقع: {str(e)}")
        context.user_data.clear()


async def check_arab_payment_status(context: ContextTypes.DEFAULT_TYPE, transaction_id: str):
    """التحقق من حالة دفعة محددة عبر Arab Payment API."""
    url = f"{API_BASE_URL}/getPaymentStatus"
    headers = {
        "Authorization": f"Bearer {ARAB_PAYMENT_API_KEY}",
        "Content-Type": "application/json"
    }
    payload = {"transaction_id": transaction_id}
    
    try:
        async with httpx.AsyncClient(timeout=10) as client:
            response = await client.post(url, headers=headers, json=payload)
            response.raise_for_status()
            result = response.json()
            
            if result.get("success") == 1 and result.get("data"):
                payment_details = result['data'][0]
                # payment_status: 0-Waiting, 1-Received, 2-TimeOut, 3-WaitingForMoreFunds
                status_code = payment_details.get("payment_status", 0)
                return status_code, payment_details
            else:
                return -1, {"error": result.get("message", "فشل API: لا يوجد بيانات")}
                
    except httpx.TimeoutException:
        return -2, {"error": "انتهت مهلة الاتصال بالخادم"}
    except httpx.RequestError as e:
        return -2, {"error": f"خطأ في الاتصال: {str(e)}"}
    except Exception as e:
        return -2, {"error": f"خطأ غير متوقع: {str(e)}"}


async def background_arab_payment_check(update, context, transaction_id, max_attempts=50, interval=10):
    """فحص حالة الدفع في الخلفية"""
    attempt = 0
    
    while attempt < max_attempts:
        attempt += 1
        status, details = await check_arab_payment_status(context, transaction_id)

        # حفظ الحالة مؤقتاً في user_data
        context.user_data["last_payment_status"] = {
            "transaction_id": transaction_id,
            "status": status,
            "details": details,
        }

        if status == 1:  # الدفع ناجح
            context.user_data['arab_pay_transaction_id'] = transaction_id
            context.user_data['deposit_method'] = 'usdt'
            await process_arab_pay_success(update, context, transaction_id, details)
            return

        elif status == 0:  # قيد الانتظار
            # تحديث حالة الانتظار للمستخدم
            if attempt % 5 == 0:  # كل 5 محاولات (50 ثانية)
                waiting_msg = (
                    f"⏳ جاري التحقق من حالة الدفع...\n"
                    f"المحاولة: {attempt}/{max_attempts}\n"
                    f"رقم المعاملة: <code>{transaction_id}</code>"
                )
                try:
                    await update.effective_message.edit_text(waiting_msg, parse_mode='HTML')
                except:
                    pass
            
            await asyncio.sleep(interval)
            continue

        else:  # فشل أو انتهاء الفاتورة أو خطأ
            error_msg = details.get('error', 'فشل في العملية أو انتهت مدة الفاتورة.')
            await update.effective_message.edit_text(f"❌ فشل الإيداع: {error_msg}", parse_mode='HTML')
            context.user_data.clear()
            return

    await update.effective_message.edit_text(
        "❌ لم يتم تلقي الدفع خلال الوقت المحدد. يرجى المحاولة مجددًا.",
        parse_mode='HTML'
    )
    context.user_data.clear()


async def check_arab_pay_callback(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """معالجة زر التحقق من حالة الدفع"""
    query = update.callback_query
    await query.answer()

    data = query.data
    if not data.startswith("check_arab_pay:"):
        return

    transaction_id = data.split(":")[1]
    user = update.effective_user

    await query.edit_message_text("⏳ جاري التحقق من حالة الدفع تلقائيًا...", parse_mode="HTML")

    # إطلاق التحقق في الخلفية
    asyncio.create_task(background_arab_payment_check(update, context, transaction_id))


async def handle_arab_pay_currency(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """معالجة اختيار عملة USDT"""
    query = update.callback_query
    await query.answer()

    data = query.data
    if not data.startswith("usdt_arab_pay_id_"):
        return ConversationHandler.END
    
    try:
        currency_id = int(data.replace("usdt_arab_pay_id_", ""))
    except ValueError:
        await query.edit_message_text("❌ معرف العملة غير صالح.")
        return ConversationHandler.END

    available = context.user_data.get("arab_pay_currencies", [])
    
    selected = next(
        (c for c in available if c.get("currency_network_id") == currency_id),
        None
    )

    if not selected:
        await query.edit_message_text("❌ العملة غير موجودة.")
        return ConversationHandler.END

    # حفظ اختيار المستخدم
    context.user_data["deposit_method"] = "usdt"
    context.user_data["arab_pay_currency_id"] = currency_id
    context.user_data["currency_display"] = f"{selected.get('currency', 'USDT')}/{selected.get('network', 'Network')}"

    # جلب الحد الأدنى للإيداع
    cfg = get_config()
    min_deposit = cfg.get("MINIMUM_DEPOSIT", {}).get("USDT", 1.0)

    text = (
        f"💰 اخترت الإيداع عبر <b>{context.user_data['currency_display']}</b>.\n"
        f"🔸 <b>الحد الأدنى للإيداع:</b> <code>{min_deposit} USDT</code>\n\n"
        f"📩 الرجاء إدخال المبلغ الذي تريد إيداعه (بالـ USDT)."
    )

    await query.edit_message_text(text, parse_mode='HTML')
    return AWAITING_DEPOSIT_AMOUNT


async def create_arab_pay_invoice(update: Update, context: ContextTypes.DEFAULT_TYPE, amount: float):
    """إنشاء فاتورة دفع عبر Arab Pay API"""
    if "arab_pay_currency_id" not in context.user_data:
        return None, "❌ لم يتم اختيار عملة للدفع."
    
    currency_id = context.user_data["arab_pay_currency_id"]
    
    url = f"{API_BASE_URL}/createInvoice"
    headers = {
        "Authorization": f"Bearer {ARAB_PAYMENT_API_KEY}",
        "Content-Type": "application/json"
    }
    
    payload = {
        "currency_network_id": currency_id,
        "amount": amount,
        "order_id": f"deposit_{int(time.time())}_{update.effective_user.id}",
        "callback_url": "",  # يمكن تركها فارغة إذا كنت تستخدم webhook
        "success_url": "",
        "cancel_url": ""
    }
    
    try:
        async with httpx.AsyncClient(timeout=10) as client:
            response = await client.post(url, headers=headers, json=payload)
            response.raise_for_status()
            result = response.json()
            
            if result.get("success") == 1 and result.get("data"):
                invoice_data = result["data"]
                return invoice_data, None
            else:
                return None, result.get("message", "فشل في إنشاء الفاتورة.")
                
    except httpx.TimeoutException:
        return None, "انتهت مهلة الاتصال بالخادم"
    except httpx.RequestError as e:
        return None, f"خطأ في الاتصال: {str(e)}"
    except Exception as e:
        return None, f"خطأ غير متوقع: {str(e)}"