import httpx
import asyncio
from threading import Thread
from database import (
    check_used_transaction,
    add_used_transaction,
    update_user_balance,
    add_transaction_log,
    add_admin_history,
    _get_setting
)
from config import SHAMCASH_API_KEY, SHAMCASH_ACCOUNT_ADDRESS, SHAMCASH_API_URL
from handlers.deposit.send_notification import send_deposit_notification


async def check_chamcash_via_api(user_id_str, client_number, expected_amount, currency):
    """
    التحقق من عملية شام كاش باستخدام API مباشرة
    """
    # التحقق من إذا كان رقم العملية مستخدمًا مسبقًا
    if await check_used_transaction(client_number):
        return "ALREADY_USED", 0
    
    # استخدام المتغيرات من config
    account_address = SHAMCASH_ACCOUNT_ADDRESS
    api_key = SHAMCASH_API_KEY
    
    # بناء URL API
    params = {
        "resource": "shamcash",
        "action": "logs",
        "account_address": account_address,
        "api_key": api_key
    }
    
    try:
        # جلب سجل التحويلات
        async with httpx.AsyncClient(timeout=30) as client:
            response = await client.get(SHAMCASH_API_URL, params=params)
            response.raise_for_status()
            data = response.json()
            
            if not data.get("success"):
                print(f"❌ استجابة API غير ناجحة: {data}")
                return "API_ERROR", 0
            
            # البحث عن العملية المطابقة
            items = data.get("data", {}).get("items", [])
            
            if not items:
                return "NOT_FOUND", 0
            
            print(f"🔍 تم العثور على {len(items)} عملية في السجل")
            
            for item in items:
                # تحقق من وجود tran_id
                tran_id = item.get("tran_id")
                if not tran_id:
                    continue
                
                # التحقق من رقم العملية
                if str(tran_id).strip() == str(client_number).strip():
                    print(f"✅ تم العثور على العملية المطابقة: {tran_id}")
                    
                    # التحقق من المبلغ
                    amount_text = str(item.get("amount", "0")).replace(",", "").replace(" ", "")
                    try:
                        extracted_amount = float(amount_text)
                    except ValueError:
                        print(f"⚠️ خطأ في تحويل المبلغ: {amount_text}")
                        continue
                    
                    # التحقق من نوع العملة
                    item_currency = str(item.get("currency", "")).upper()
                    print(f"💰 العملة في السجل: {item_currency}, العملة المطلوبة: {currency}")
                    
                    extracted_amount_ls = 0
                    
                    if currency.upper() == "USD":
                        if item_currency in ["USD", "$", "دولار", "DOLLAR"]:
                            # تحويل USD إلى SYP
                            exchange_rate = float(_get_setting("exchange_rate", 4500))
                            extracted_amount_ls = extracted_amount * exchange_rate
                            print(f"💱 تحويل USD إلى SYP: {extracted_amount} USD × {exchange_rate} = {extracted_amount_ls} SYP")
                        else:
                            # العملة غير مطابقة
                            print(f"⚠️ العملة غير مطابقة: {item_currency} != USD")
                            continue
                    elif currency.upper() in ["SYP", "LS", "ليرة", "ل.س"]:
                        if item_currency in ["SYP", "LS", "ليرة", "ل.س", "LIRA"]:
                            
                            extracted_amount_ls = extracted_amount
                        else:
                            # العملة غير مطابقة
                            print(f"⚠️ العملة غير مطابقة: {item_currency} != SYP")
                            continue
                    
                    # التحقق من المبلغ المتوقع
                    expected_amount_float = float(expected_amount)
                    
                    print(f"🔢 مقارنة المبالغ - المستخرج: {extracted_amount}, المتوقع: {expected_amount_float}")
                    
                    if currency.upper() == "USD":
                        # إذا كانت العملة USD، قارن مباشرة
                        if abs(extracted_amount - expected_amount_float) > 0:  # هامش خطأ 0.1
                            print(f"❌ عدم تطابق المبلغ USD: {extracted_amount} != {expected_amount_float}")
                            return "AMOUNT_MISMATCH", 0
                    else:
                        # إذا كانت العملة SYP، قارن مباشرة
                        if abs(extracted_amount - expected_amount_float) > 0:  # هامش خطأ 100 ليرة
                            print(f"❌ عدم تطابق المبلغ SYP: {extracted_amount} != {expected_amount_float}")
                            return "AMOUNT_MISMATCH", 0
                    
                    final_amount_ls = extracted_amount_ls

                    # إضافة 00 فقط إذا كانت ليرة سورية
                    if currency.upper() in ["SYP", "LS", "ليرة", "ل.س"]:
                        final_amount_ls *= 100
                        print(f"➕ تمت إضافة 00 للمبلغ: {extracted_amount_ls} → {final_amount_ls}")
                    # ✅ تسجيل العملية كمستخدمة
                    await add_used_transaction(
                        reference_id=client_number,
                        amount=final_amount_ls,
                        method="cham_cash",
                        user_id=user_id_str
                    )
                    
                    return "SUCCESS", final_amount_ls
            
            # إذا وصلنا هنا ولم نجد العملية
            print(f"❌ لم يتم العثور على العملية {client_number} في السجل")
            return "NOT_FOUND", 0
            
    except httpx.TimeoutException:
        print(f"❌ انتهت مهلة الاتصال بـ API شام كاش")
        return "API_TIMEOUT", 0
    except httpx.RequestError as e:
        print(f"❌ خطأ في الاتصال بـ API شام كاش: {e}")
        return "API_CONNECTION_ERROR", 0
    except Exception as e:
        print(f"❌ خطأ غير متوقع في API شام كاش: {e}")
        return "ERROR", 0

async def process_chamcash_deposit(user_id_str, client_number, amount_input, currency, bot, context, user):
    """
    معالجة إيداع شام كاش كاملة
    """
    from config import get_config
    
    # استخدام API للتحقق
    verification_status, amount_ls = await check_chamcash_via_api(
        user_id_str, client_number, amount_input, currency
    )
    
    final_message = ""
    notification_status = ""
    notification_details = ""
    
    if verification_status == "SUCCESS":
        
        cfg = get_config()
        bonus_amount = 0.0
        bonus_percentage = cfg.get("Bouns_Gift", {}).get("SHAM", 0)
        if bonus_percentage > 0:
            bonus_amount = (amount_ls * bonus_percentage) / 100.0
        
        total_amount = amount_ls + bonus_amount
        
        # ✅ تحديث رصيد المستخدم
        await update_user_balance(user_id_str, total_amount, amount_ls)
        
        # ✅ تسجيل العملية
        await add_transaction_log(
            user_id_str, 'deposit', 'CHAM_CASH',
            amount_ls, bonus_amount, client_number
        )
        
        # ✅ تسجيل في سجلات الأدمن
        add_admin_history(
            user_id_str, "deposit", amount_ls, "CHAM_CASH", 
            status="success", extra=f"client_num={client_number}"
        )
        deposit_tx = {
            "transaction_id": client_number,
            "amount": float(amount_ls)
        }

        # ✅ معالجة الإحالة (النظام الجديد)
        try:
            from handlers.referral.referral_service import process_referral_for_deposit

            await process_referral_for_deposit(
                invited_id=user_id_str,
                deposit_tx=deposit_tx,
                bot=bot
            )

        except Exception as e:
            print(f"⚠️ فشل معالجة الإحالة (CHAM CASH): {e}")


        
        final_message = (
            f"✅ تم تأكيد إيداعك بنجاح!\n\n"
            f"💰 المبلغ الأساسي: <code>{int(amount_ls):,}</code> ل.س\n"
        )
        if bonus_amount > 0:
            final_message += f"🎁 البونص: <code>{int(bonus_amount):,}</code> ل.س ({bonus_percentage:.1f}%)\n"
        final_message += f"💵 المجموع الكلي: <code>{int(total_amount):,}</code> ل.س"
        
        notification_status = "SUCCESS"
        
    elif verification_status == "ALREADY_USED":
        final_message = "❌ رقم العملية هذا تم استخدامه سابقًا."
        notification_status = "DUPLICATE"
        
    elif verification_status in ["NOT_FOUND", "AMOUNT_MISMATCH", "API_ERROR", "API_CONNECTION_ERROR", "API_TIMEOUT"]:
        error_messages = {
            "NOT_FOUND": "❌ لم يتم العثور على العملية في سجلات شام كاش.",
            "AMOUNT_MISMATCH": "❌ المبلغ في العملية لا يتطابق مع المبلغ المدخل.",
            "API_ERROR": "❌ حدث خطأ في استجابة API شام كاش.",
            "API_CONNECTION_ERROR": "❌ تعذر الاتصال بخدمة شام كاش.",
            "API_TIMEOUT": "❌ انتهت مهلة الاتصال بخدمة شام كاش."
        }
        final_message = error_messages.get(verification_status, "❌ فشل التحقق من العملية.")
        notification_status = "FAILED"
        notification_details = final_message
        
    else:
        final_message = "❌ حدث خطأ فني أثناء التحقق من العملية. يرجى مراجعة الدعم."
        notification_status = "FAILED"
        notification_details = final_message
    
    # إرسال إشعار القناة
    if notification_status:
        try:
            await send_deposit_notification(
                context=context,
                user=user,
                status=notification_status,
                amount=amount_ls if verification_status == "SUCCESS" else 0,
                method="cham_cash",
                client_number=client_number,
                details=notification_details
            )
        except Exception as e:
            print(f"⚠️ خطأ في إرسال إشعار القناة: {e}")
    
    # إرسال النتيجة للمستخدم
    try:
        await bot.send_message(chat_id=user_id_str, text=final_message, parse_mode="HTML")
    except Exception as e:
        print(f"❌ خطأ في إرسال الرسالة للمستخدم: {e}")

def run_auto_deposit_check(user_id_str, client_number, amount_input, currency, bot, context, user, loop):
    """
    تشغيل عملية التحقق من خلال thread
    """
    async def wrapper():
        await process_chamcash_deposit(
            user_id_str, client_number, amount_input, currency, 
            bot, context, user
        )
    
    # تشغيل الدالة غير المتزامنة في الـ loop المحدد
    asyncio.run_coroutine_threadsafe(wrapper(), loop)